#!/usr/bin/python3.10

import collections
import datetime
import json
import math
import os
import re
import sys
import traceback

import ephem
import matplotlib.font_manager
import matplotlib.pyplot as plt
import numpy as np
import pandas as pd
from matplotlib.colors import from_levels_and_colors

matplotlib.use('TkAgg')


def resource_path(relative_path):
    """Necessary when creating *.exe with pyinstaller."""
    try:
        base_path = sys._MEIPASS
    except Exception:
        base_path = os.path.abspath(".")

    return os.path.join(base_path, relative_path)


def replace_nun_alphanumeric_characters(txt):
    return re.sub("[^0-9a-zA-Z\-_\.]+", "_", txt)


class CometAltitudeChart:
    def __init__(
            self,
            *,
            date_start_utc,
            lat,
            lon,
            observing_site_name,
            comet_csv_data_line

    ):
        self.hour_utc_of_col_0 = 12  # 12:00 utc
        self.date_start_utc = datetime.datetime(date_start_utc.year, date_start_utc.month, 1, self.hour_utc_of_col_0,
                                                0)  # 22:00 UTC [!]  ==  00 MESZ

        self.observing_site = ephem.Observer()
        self.observing_site.lat, self.observing_site.lon = f"{lat:.10f}", f"{lon:.10f}"
        self.observing_site.horizon = '-0:00'
        self.observing_site.name = observing_site_name

        # Comet's data line from https://minorplanetcenter.net/iau/Ephemerides/Comets/Soft03Cmt.txt
        # comet_csv_data_line = "12P/Pons-Brooks,e,74.1914,255.8560,198.9895,17.20432,0.0138117,0.95461748,359.4458,03/12.0/2024,2000,g  5.0,6.0"
        self.comet = ephem.readdb(comet_csv_data_line)

    def calc_alt_deg_comet(self, dt_obj, delta_minutes):
        self.observing_site.date = ephem.Date(dt_obj + datetime.timedelta(minutes=delta_minutes))  # 23 UTC = 0 CET
        self.comet.compute(self.observing_site)
        return self.comet.alt * 180 / math.pi

    def calc_brightness_mag_comet(self, dt_obj, delta_minutes=0):
        self.observing_site.date = ephem.Date(dt_obj + datetime.timedelta(minutes=delta_minutes))  # 23 UTC = 0 CET
        self.comet.compute(self.observing_site)
        return self.comet.mag

    def calc_alt_deg_sun(self, dt_obj, delta_minutes):
        self.observing_site.date = ephem.Date(dt_obj + datetime.timedelta(minutes=delta_minutes))  # 23 UTC = 0 CET
        return ephem.Sun(self.observing_site).alt * 180 / math.pi

    def calc_alt_deg_moon(self, dt_obj, delta_minutes):
        self.observing_site.date = ephem.Date(dt_obj + datetime.timedelta(minutes=delta_minutes))  # 23 UTC = 0 CET
        return ephem.Moon(self.observing_site).alt * 180 / math.pi

    def calculate_ephemeris_for_time_range(self):
        # Only plot night time values from 12+2 UTC = 14 UTC = 15 CET until 12+24-4 UTC = 32 UTC = 8 UTC = 9 CET
        self.columns_range_minutes = range((0 + 2) * 60, (24 - 4) * 60 + 1)

        num_months = 2
        date_end_utc = pd.Timestamp(self.date_start_utc) + pd.DateOffset(months=num_months)
        self.amount_days = (date_end_utc.to_pydatetime() - self.date_start_utc).days

        if True and "Dump df to disk":
            df_index = pd.date_range(
                self.date_start_utc,
                # periods=60*24*30*3,
                # freq=datetime.timedelta(minutes=1)

                # periods=24*30*3,
                # freq=datetime.timedelta(hours=1),

                periods=self.amount_days * 1,
                freq=datetime.timedelta(hours=24)
            )
            # df_index.set_names('time_utc')

            df_columns = list(self.columns_range_minutes)

            print("\t" * 3 + "Calculating comet ephemeris.")
            lst_comet_alt_deg = [
                [self.calc_alt_deg_comet(date_obj, delta_min) for delta_min in self.columns_range_minutes] for
                date_obj in df_index]
            self.df_comet_alt_deg = pd.DataFrame(index=df_index, columns=df_columns, data=lst_comet_alt_deg)

            lst_comet_brightness_mag = [self.calc_brightness_mag_comet(date_obj) for date_obj in df_index]
            self.df_comet_brightness_mag = pd.DataFrame(index=df_index, columns=["mag"], data=lst_comet_brightness_mag)

            print("\t" * 3 + "Calculating moon ephemeris.")
            lst_moon_alt_deg = [
                [self.calc_alt_deg_moon(date_obj, delta_min) for delta_min in self.columns_range_minutes] for
                date_obj
                in df_index]
            self.df_moon_alt_deg = pd.DataFrame(index=df_index, columns=df_columns, data=lst_moon_alt_deg)

            print("\t" * 3 + "Calculating sun ephemeris.")
            lst_sun_alt_deg = [[self.calc_alt_deg_sun(date_obj, delta_min) for delta_min in self.columns_range_minutes]
                               for
                               date_obj
                               in df_index]
            self.df_sun_alt_deg = pd.DataFrame(index=df_index, columns=df_columns, data=lst_sun_alt_deg)

            if False:
                self.df_comet_alt_deg.to_pickle("df_comet_alt_deg.pandas_df")
                self.df_moon_alt_deg.to_pickle("df_moon_alt_deg.pandas_df")
                self.df_sun_alt_deg.to_pickle("df_sun_alt_deg.pandas_df")
                self.df_comet_brightness_mag.to_pickle("df_comet_brightness_mag.pandas_df")
        else:
            self.df_comet_alt_deg = pd.read_pickle("df_comet_alt_deg.pandas_df")
            self.df_moon_alt_deg = pd.read_pickle("df_moon_alt_deg.pandas_df")
            self.df_sun_alt_deg = pd.read_pickle("df_sun_alt_deg.pandas_df")
            self.df_comet_brightness_mag = pd.read_pickle("df_comet_brightness_mag.pandas_df")

    def configure_plot_and_write_png(self):
        sub_plots_tpl = plt.subplots()
        fig: matplotlib.figure.Figure = sub_plots_tpl[0]
        ax: matplotlib.axes._axes.Axes = sub_plots_tpl[1]

        fig.set_size_inches(21, 29.7)
        fig.subplots_adjust(
            left=1 / 21 + 3 / 21,
            # The position of the left edge of the subplots, as a fraction of the figure width.
            right=20 / 21,
            bottom=1 / 29.7,
            top=(29.7 - 2) / 29.7,
        )

        plt.title("{} at {} N{:.1f}° E{:.1f}° in {}-{}".format(
            self.comet.name,
            self.observing_site.name,
            self.observing_site.lat / math.pi * 180,
            self.observing_site.lon / math.pi * 180,
            self.date_start_utc.year,
            self.date_start_utc.month,

        ), fontsize=30, fontweight="bold", pad=20)

        x = pd.Series([i for i in self.columns_range_minutes])
        y = self.df_sun_alt_deg.index

        if "Comet (color mesh)":
            z_values_comet = self.df_comet_alt_deg
            levels_comet = [-90, 0, 5, 10, 15, 20, 30, 40, 50, 60, 70, 80, 90]

            # https://colordesigner.io/gradient-generator
            # F84F4F (red) to 39EA5F (green), 20 steps
            """
                        #f84f4f
                        #f85945
                        #f7623b
                        #f46c31
                        #f17526
                        #ed7f18
                        #e88803
                        #e29100
                        #db9a00
                        #d3a300
                        #cbab00
                        #c1b300
                        #b7bb00
                        #abc300
                        #9fca00
                        #91d111
                        #81d829
                        #6fde3c
                        #59e44d
                        #39ea5f
                        """

            # #d8ff4d (green) to #39ea5f (green), 9 steps
            # #d8ff4d
            # #c8fd4c
            # #b8fb4d
            # #a8f84e
            # #96f650
            # #84f353
            # #6ff056
            # #58ed5a
            # #39ea5f

            colors_comet = [
                "midnightblue",  # -90 .. 0
                "#d3a300",  # 0 .. 5° (red)
                "#cbab00",  # 5 .. 10° (red)

                "#d8ff4d",  # 10 .. 15° (green)
                "#c8fd4c",  # 15 .. 20°
                "#b8fb4d",  # 20 .. 30°
                "#a8f84e",  # 30 .. 40°
                "#96f650",  # 40 .. 50°
                "#84f353",  # 50 .. 60°
                "#6ff056",  # 60 .. 70°
                "#58ed5a",  # 70 .. 80°
                "#39ea5f",  # 80 .. 90° (green)
            ]

            cmap_comet, norm_comet = from_levels_and_colors(
                levels_comet,
                colors_comet
            )  # mention levels and colors here

            color_mesh_comet = ax.pcolormesh(
                x, y, z_values_comet,
                cmap=cmap_comet,
                norm=norm_comet
            )

        if "Comet (contour lines)":
            CL_comet = ax.contour(
                x, y, z_values_comet,
                levels_comet,
                colors="black",
                linewidths=0.25,
                linestyles="solid"
            )  # Negative contours default to dashed.
            clabels_comet = ax.clabel(CL_comet, fontsize=20, inline=True, fmt="%1.0f°")

            for cl in clabels_comet:
                if cl.get_rotation() < 90:
                    cl.set_rotation(-1 * cl.get_rotation())
                else:
                    cl.set_rotation(-1 * cl.get_rotation())
                    a = 0

        if "Sun (filled contour - daylight only)":
            # z_values_sun = df_sun_alt_deg.iloc[0:, 1:]
            z_values_sun = self.df_sun_alt_deg
            levels_sun = [-90, -18, -12, -6, 0, 90]

            colors_sun = (
                "red",  # Astronomical darkness (disabled)
                "red",  # Astronomical twilight (disabled)
                "red",  # Nautical twilight (disabled)
                "red",  # Civil twilight (disabled)
                "skyblue",  # Daylight
            )
            cmap, norm = from_levels_and_colors(levels_sun, colors_sun)  # mention levels and colors here

            color_mesh_daylight = ax.pcolormesh(
                x, y, np.where(z_values_sun < 0, np.nan, z_values_sun),  # hide ranges where sun < 0°
                cmap=cmap, norm=norm,
                alpha=1,
            )

        if "Sun (filled contour - civil twilight only)":
            # z_values_sun = df_sun_alt_deg.iloc[0:, 1:]
            z_values_sun = self.df_sun_alt_deg
            levels_sun = [-90, -18, -12, -6, 0, 90]

            colors_sun = (
                "red",  # Astronomical darkness (disabled)
                "red",  # Astronomical twilight (disabled)
                "red",  # Nautical twilight (disabled)
                "#e2e2e2",  # Civil twilight
                "red",  # Daylight (disabled)
            )

            cmap, norm = from_levels_and_colors(levels_sun, colors_sun)  # mention levels and colors here

            color_mesh_civil_twilight = ax.pcolormesh(
                # x, y, px_values,
                x, y, np.where((z_values_sun < -6) | (z_values_sun > 0), np.nan, z_values_sun),
                # hide range < -6° and > 0°
                cmap=cmap, norm=norm,
                alpha=0.75,
            )
            # ax.set_aspect('equal')
            # ax.clabel(CS, inline=1, fontsize=10)

        if "Sun (filled contour - nautical twilight only)":
            # z_values_sun = df_sun_alt_deg.iloc[0:, 1:]
            z_values_sun = self.df_sun_alt_deg
            levels_sun = [-90, -18, -12, -6, 0, 90]

            colors_sun = (
                "red",  # Astronomical darkness (disabled)
                "red",  # Astronomical twilight (disabled)
                "#e2e2e2",  # Nautical twilight
                "red",  # Civil twilight (disabled)
                "red",  # Daylight (disabled)
            )

            cmap, norm = from_levels_and_colors(levels_sun, colors_sun)  # mention levels and colors here

            color_mesh_nautical_twiligh = ax.pcolormesh(
                # x, y, px_values,
                x, y, np.where((z_values_sun < -12) | (z_values_sun > -6), np.nan, z_values_sun),
                # hide range < -12° and > 6°
                cmap=cmap, norm=norm,
                alpha=0.5,
            )
            # ax.set_aspect('equal')
            # ax.clabel(CS, inline=1, fontsize=10)

        if "Sun (contour lines)":
            CL_sun = ax.contour(x, y, z_values_sun, levels_sun, colors="bisque",
                                linewidths=1,
                                linestyles="solid")
            clabels_sun = ax.clabel(CL_sun, fontsize=20, inline=True, fmt="Sun %1.0f°")
            for cl in clabels_sun:
                if cl.get_rotation() < 180:
                    cl.set_rotation(-1 * cl.get_rotation())
                else:
                    cl.set_rotation(-1 * cl.get_rotation())

        if "Moon":
            z_values_moon = self.df_moon_alt_deg

            levels_moon = [-90, 0, 90]
            colors_moon = (
                'black',
                'bisque'  # FFE4C4
            )
            cmap_moon, norm_moon = from_levels_and_colors(levels_moon, colors_moon)

            color_mesh_moon = ax.pcolormesh(
                x, y, np.where(z_values_moon < 0, np.nan, z_values_moon),
                cmap=cmap_moon, norm=norm_moon,
                alpha=0.5,
            )
            ax.set_aspect('auto')

        if "Moon (contour lines)":
            CL_moon = ax.contour(x, y, z_values_moon, levels_moon,
                                 colors="bisque",
                                 linewidths=1,
                                 linestyles="solid")
            clabels_moon = ax.clabel(CL_moon, fontsize=20, inline=True, fmt="Moon %1.0f°")
            for cl in clabels_moon:
                cl.set_rotation(-1 * cl.get_rotation())

        if "Plot moon phases":
            moon_phases = collections.defaultdict(list)

            for next_fn in [
                ephem.next_new_moon,
                ephem.next_first_quarter_moon,
                ephem.next_full_moon,
                ephem.next_last_quarter_moon,
            ]:
                current_day = self.date_start_utc
                while current_day <= self.date_start_utc + datetime.timedelta(days=self.amount_days):
                    current_day = next_fn(current_day).datetime()
                    if current_day <= (self.date_start_utc + datetime.timedelta(days=self.amount_days)).replace(hour=0):
                        moon_phases[next_fn].append(current_day)

            f0 = matplotlib.font_manager.FontProperties()
            f0.set_file(
                resource_path("font/DejaVuSans.ttf")
            )

            for moon_phase, list_of_dates in moon_phases.items():
                for dt_obj in list_of_dates:
                    plt.text(
                        int(3 * 60 + 10),
                        datetime.datetime(year=dt_obj.year, month=dt_obj.month, day=dt_obj.day, hour=12),
                        {
                            ephem.next_new_moon: u'\u25CF',
                            ephem.next_first_quarter_moon: u'\u25D0',
                            ephem.next_full_moon: u'\u25CB',
                            ephem.next_last_quarter_moon: u'\u25D1',
                        }[moon_phase],
                        fontproperties=f0,
                        rotation=0 * (180 / math.pi),
                        size=30,
                        va='center',
                    )

        if "Print timezone: CET or CEST as x axis label":
            displayed_months = set([itm.month for itm in self.df_comet_alt_deg.index])
            months_with_mainly_winter_time = set([1, 2, 3, 11, 12])
            if displayed_months.intersection(months_with_mainly_winter_time).__len__() == 0:
                timezone = "CEST"
                tz_hours = 2
            else:
                timezone = "CET"
                tz_hours = 1

            ax.set_xlabel(timezone, fontsize=20)

        custom_x_axis_ticks_and_labels = {
            i * 60: "{}".format((self.hour_utc_of_col_0 + i + tz_hours) % 24) for i in
            range(int(self.columns_range_minutes.start / 60), int((self.columns_range_minutes.stop - 1) / 60) + 1)
        }

        ax.set_xticks(list(custom_x_axis_ticks_and_labels.keys()))
        ax.set_xticklabels(list(custom_x_axis_ticks_and_labels.values()))

        ax.tick_params(axis="x", bottom=True, top=True, labelbottom=True, labeltop=True)

        ax.tick_params(axis='both', which='major', labelsize=20)

        ax.invert_yaxis()
        major_y_ticks = self.df_comet_alt_deg.index.to_list()
        minor_y_ticks = [itm + datetime.timedelta(hours=12) for itm in major_y_ticks]
        ax.set_yticks(major_y_ticks)
        ax.set_yticks(minor_y_ticks, minor=True)

        if "Left side y tick labels (comet brightness [mag] and date)":
            ax.set_yticklabels(["{:.1f} m  {}".format(tpl[1], tpl[0].strftime("%Y-%m-%d")) for tpl in
                                zip(self.df_comet_alt_deg.index.to_list(),
                                    self.df_comet_brightness_mag["mag"].to_list())])

        if False and "Right side y tick labels (next day)":
            ax2 = ax.secondary_yaxis('right')

            major_y_secondary_ticks__dict = {
                day: (day + datetime.timedelta(hours=24)).strftime("%Y-%m-%d") for day in
                self.df_comet_alt_deg.index.to_list()
            }

            ax2.set_yticks(list(major_y_secondary_ticks__dict.keys()))
            ax2.set_yticklabels(list(major_y_secondary_ticks__dict.values()))

            # Change font size of x and y ticks
            ax2.tick_params(axis='both', which='major', labelsize=20)

        # https://matplotlib.org/stable/gallery/color/named_colors.html
        ax.grid(axis='y', which="minor", color="darkgray")
        ax.grid(axis='x', which="major", color="darkgray")

        filename_png = "Altitude_chart__{}__{}__{}-{}.png".format(
            replace_nun_alphanumeric_characters(self.comet.name),
            replace_nun_alphanumeric_characters(self.observing_site.name),
            self.date_start_utc.year,
            self.date_start_utc.month,
        )
        filename_png = re.sub("__+", "__", filename_png)

        print("\t" * 3 + f"Writing {filename_png}.")
        plt.savefig(
            filename_png,
            dpi=100,
            pad_inches=0.4
        )
        print("\t" * 3 + f"{filename_png} written.")
        # plt.show()

    def test_sunset(self):
        self.observing_site.date = ephem.Date(datetime.datetime(2024, 12, 1, 12, 0))  # 23 UTC = 0 CET
        sun = ephem.Sun(self.observing_site)
        print(self.observing_site.next_setting(sun))  # 2024/12/1 15:22:20
        """
        Für den 01.12.2024 gilt für die ausgewählte Position:
        Sonnenaufgang: 08:05
        Sonnenuntergang: 16:18
        """


if __name__ == "__main__":
    try:
        with open("config.json", "r") as fh:
            config = json.loads(fh.read())

        with open("comet_data_from_minorplanetcenter_Soft03Cmt.txt", "r") as fh:
            comet_data = [line.strip() for line in fh.read().split("\n")]
            comet_data = list(filter(lambda x: not x.startswith("#"), comet_data))

        for i_location, location in enumerate(config['locations'], 1):
            print(f"Processing location {i_location} of {config['locations'].__len__()}: {location}")
            for i_comet_data_line, comet_data_line in enumerate(comet_data, 1):
                print(f"\tProcessing comet {i_comet_data_line} of {comet_data.__len__()}: {comet_data_line.split(',')[0]}")
                start_yr_1st_chart, start_mo_1st_chart = map(int, config['start_date_year_and_month'].split("-"))
                start_date_first_chart = datetime.datetime(start_yr_1st_chart, start_mo_1st_chart, 1)

                # 1 chart per 2 months:
                start_dates_for_each_chart = [pd.Timestamp(start_date_first_chart) + pd.DateOffset(months=i * 2) for i in
                                              range(math.ceil(config['forecast_range_months'] / 2))]
                start_dates_for_each_chart = [itm.to_pydatetime() for itm in start_dates_for_each_chart]

                for i_start_date, start_date in enumerate(start_dates_for_each_chart, 1):
                    print(
                        f"\t\tProcessing date range {i_start_date} of {start_dates_for_each_chart.__len__()}: {start_date}...")
                    ephemeris_and_chart_for_2_months = CometAltitudeChart(
                        date_start_utc=start_date,
                        lat=location['lat'],
                        lon=location['lon'],
                        observing_site_name=location['title'],
                        comet_csv_data_line=comet_data_line,
                    )
                    ephemeris_and_chart_for_2_months.calculate_ephemeris_for_time_range()
                    ephemeris_and_chart_for_2_months.configure_plot_and_write_png()
    except Exception as e:
        traceback.print_exc()

    input("Press any key to exit the console application.")